#include <nds.h>
#include <stdlib.h>
#include <nds/arm9/trig_lut.h>

#include "title_bin.h"
#include "title_sub_bin.h"
#include "load_screen_bin.h"
#include "edit_screen_bin.h"
#include "show_screen_bin.h"

#include "ship_bin.h"
#include "ship2_bin.h"
#include "bad_bin.h"
#include "laser_bin.h"

typedef struct {
	int x1, y1, x2, y2;
	int w; //width
	u16 c1, c2;
} edit_box;

typedef struct {
	int x, y;
	int s;
	u16 c;
} shiny_star;

typedef struct {
	int x1, y1, x2, y2, dx, dy;
	int size; //8, 16, or 32
	int id, gfx; //offset in memory
	int fx, fy; //gun location
	int s; //ds screen
	int a; //alive?
	SpriteEntry* oam;
	SpriteRotation* rot;
} sprite;

void swap(int* a, int* b); 
void init_oam();
void init_hardware();
void update_oam();

void main_menu();
int load_sprite(int cur_sprite);

void edit_loop(int cur_sprite);
	void draw_obj(int cur_sprite);
	void draw_box(edit_box* box);
	int is_in(int x, int y, edit_box* box);

void show_sprite(int cur_sprite);

void play_game();
	int collide(sprite* one, sprite* two);
	
void play_game2();

SpriteEntry oam_copy[128];
SpriteEntry oam_copy_sub[128];
SpriteRotation* sprite_rotation = (SpriteRotation*)oam_copy;
SpriteRotation* sprite_rotation_sub = (SpriteRotation*)oam_copy_sub;

const u16 WHITE = RGB15(31,31,31) | (1<<15);
const u16 BLACK = RGB15(0,0,0) | (1<<15);
const u16 RED = RGB15(31,0,0) | (1<<15);
const u16 GREEN = RGB15(0,31,0) | (1<<15);
const u16 BLUE = RGB15(0,0,31) | (1<<15);
const u16 CLEAR = !(1<<15);

const int NUM_SIZES = 3;
const int NUM_COLORS = 10;
const int NUM_USER_SPRITES = 7;
const int NUM_SPRITES = 32;
const int NUM_STARS = 100;
sprite sprites[32];

touchPosition touch;

//---------------------------------------------------------------------------------
int main(void) {
//---------------------------------------------------------------------------------
	init_hardware();
	
	int i;
	
	for (i = 0; i < NUM_SPRITES; i++) {
		sprites[i].oam = &oam_copy[i];
		sprites[i].rot = &sprite_rotation[0];
		sprites[i].size = 32;
		sprites[i].gfx = i*32*32*2;
	}
	
	for (i = 0; i < 32*32*2*NUM_SPRITES; i++) {
		SPRITE_GFX[i] = BLACK;
		SPRITE_GFX_SUB[i] = BLACK;
	}
	
	main_menu();
	
	return 0;
}

void swap(int* a, int* b) {
	int temp = *a;
	*a = *b;
	*b = temp;
}

void update_oam(void) {
	DC_FlushRange(oam_copy, 128 * sizeof(SpriteEntry));
	dmaCopy(oam_copy, OAM, 128 * sizeof(SpriteEntry));
	DC_FlushRange(oam_copy_sub, 128 * sizeof(SpriteEntry));
	dmaCopy(oam_copy_sub, OAM_SUB, 128 * sizeof(SpriteEntry));
}

void init_oam(void) {
	int i;
	for (i = 0; i < 128; i++) {
		oam_copy[i].attribute[0] = ATTR0_DISABLED;
		oam_copy_sub[i].attribute[0] = ATTR0_DISABLED;
	}
	
	update_oam();
}

void init_screen(void) {
	int i;
	for (i = 0; i < SCREEN_WIDTH * SCREEN_HEIGHT; i++) {
		BG_GFX[i] = BLACK;
		BG_GFX_SUB[i] = BLACK;
	}
}

void init_hardware(void) {
	powerON(POWER_ALL_2D);

	irqInit();
	irqEnable(IRQ_VBLANK);
	
	vramSetMainBanks(VRAM_A_MAIN_BG_0x06000000, VRAM_B_MAIN_SPRITE_0x06400000,
					 VRAM_C_SUB_BG_0x06200000, VRAM_D_SUB_SPRITE);

	videoSetMode(MODE_5_2D | DISPLAY_BG3_ACTIVE | DISPLAY_SPR_ACTIVE | DISPLAY_SPR_1D | DISPLAY_SPR_1D_BMP);
	videoSetModeSub(MODE_5_2D | DISPLAY_BG3_ACTIVE | DISPLAY_SPR_ACTIVE | DISPLAY_SPR_1D | DISPLAY_SPR_1D_BMP);

	BG3_CR = BG_BMP16_256x256;
	BG3_XDX = 1 << 8;
	BG3_XDY = 0;
	BG3_YDX = 0;
	BG3_YDY = 1 << 8;
	BG3_CX = 0;
	BG3_CY = 0;
		
	SUB_BG3_CR = BG_BMP16_256x256;
	SUB_BG3_XDX = 1 << 8;
	SUB_BG3_XDY = 0;
	SUB_BG3_YDX = 0;
	SUB_BG3_YDY = 1 << 8;
	SUB_BG3_CX = 0;
	SUB_BG3_CY = 0;
}

void main_menu(void) {
	init_oam();
	init_screen();

	int NUM_OPTIONS = 4;
	
	int i;
	int pressed = 1;
	int keys, held;
	int cur_sprite = 0;
	
	dmaCopy(title_bin, BG_GFX, SCREEN_WIDTH * SCREEN_HEIGHT * 2);
	dmaCopy(title_sub_bin, BG_GFX_SUB, SCREEN_WIDTH * SCREEN_HEIGHT * 2);

	dmaCopy(ship_bin, SPRITE_GFX, 32 * 32 * 2);
	dmaCopy(bad_bin, SPRITE_GFX + 32*32*2, 16 * 16 * 2);
	dmaCopy(bad_bin, SPRITE_GFX + 32*32*4, 16 * 16 * 2);
	dmaCopy(bad_bin, SPRITE_GFX + 32*32*6, 16 * 16 * 2);
	dmaCopy(laser_bin, SPRITE_GFX + 32*32*8, 8 * 8 * 2);
	dmaCopy(ship2_bin, SPRITE_GFX + 32 * 32 * 10, 32 * 32 * 2);
	dmaCopy(laser_bin, SPRITE_GFX + 32*32*12, 8 * 8 * 2);
	sprites[1].size = 16;
	sprites[2].size = 16;
	sprites[3].size = 16;
	sprites[4].size = 8;
	sprites[6].size = 8;
	

	
	for (i = 0; i < 32 * 32 * 2 * NUM_SPRITES; i++)
		if (SPRITE_GFX[i] == BLACK) SPRITE_GFX[i] &= CLEAR;
		
	for (i = 32 * 32 * 2; i < 32 * 32 * 4; i++)
		if (SPRITE_GFX[i] == RED) SPRITE_GFX[i] = (rand() & RGB15(31,0,0)) | (RGB15(24,0,0) | BIT(15));
	for (i = 32 * 32 * 4; i < 32 * 32 * 6; i++)
		if (SPRITE_GFX[i] == RED) SPRITE_GFX[i] = (rand() & RGB15(0,0,31)) | (RGB15(0,0,24) | BIT(15));
	for (i = 32 * 32 * 6; i < 32 * 32 * 8; i++)
		if (SPRITE_GFX[i] == RED) SPRITE_GFX[i] = (rand() & RGB15(0,31,0)) | (RGB15(0,24,0) | BIT(15));
	
	edit_box option[NUM_OPTIONS];
	
	
	for (i = 0; i < NUM_OPTIONS; i++) {
		option[i].w = 32;
		option[i].x1 = 18 + i * 62;
		option[i].y1 = SCREEN_HEIGHT / 2;
		option[i].x2 = option[i].x1 + option[i].w - 1;
		option[i].y2 = option[i].y1 + option[i].w - 1;
		option[i].c1 = WHITE;
		option[i].c2 = BLACK;
		draw_box(&option[i]);
	}

	while (1) {
		touch = touchReadXY();
		scanKeys();
		keys = keysDown();
		held = keysHeld();
		
		if ((touch.px && !pressed) || (keys)) {
			int refresh = 0;

			if (is_in(touch.px, touch.py, &option[0]) || keys & KEY_START) {
				play_game();
				refresh = 1;
			}
			if (is_in(touch.px, touch.py, &option[1])) {
				cur_sprite = load_sprite(cur_sprite);
				refresh = 1;
			}
			if (is_in(touch.px, touch.py, &option[2])) {
				edit_loop(cur_sprite);
				refresh = 1;
			}
			if (is_in(touch.px, touch.py, &option[3])) {
				show_sprite(cur_sprite);
				refresh = 1;
			}
			if ((held & KEY_L) && (held & KEY_R) && (held & KEY_DOWN) && (held & KEY_X)) {
				play_game2();
				refresh = 1;
			}

			if (refresh) {
				init_oam();
				init_screen();
				dmaCopy(title_bin, BG_GFX, SCREEN_WIDTH * SCREEN_HEIGHT * 2);
				dmaCopy(title_sub_bin, BG_GFX_SUB, SCREEN_WIDTH * SCREEN_HEIGHT * 2);
				for (i = 0; i < NUM_OPTIONS; i ++)
					draw_box(&option[i]);
				refresh = 0;
			}
		}
		
		if (touch.px)
			pressed = 1;
		else
			pressed = 0;
	}
}

void edit_loop(int cur_sprite) {
	init_oam();
	init_screen();

	int i, j;
	int pressed = 1;
	u16 sprite_buf[sprites[cur_sprite].size * sprites[cur_sprite].size];
	
	dmaCopy(edit_screen_bin, BG_GFX, SCREEN_WIDTH * SCREEN_HEIGHT * 2);
	for (i = 0; i < sprites[cur_sprite].size * sprites[cur_sprite].size; i++)
		sprite_buf[i] = SPRITE_GFX[sprites[cur_sprite].gfx + i];
	
	edit_box main_box;
	edit_box box_gfx;
	edit_box dot;
	
	edit_box scale[NUM_SIZES];
	
	edit_box colors[NUM_COLORS];
	edit_box cur_color;
	edit_box eraser;
	edit_box reraser;
	
	edit_box go;
	
	main_box.w = 128;
	main_box.x1 = (SCREEN_WIDTH - main_box.w) / 2;
	main_box.y1 = (SCREEN_HEIGHT - main_box.w) / 2;
	main_box.x2 = main_box.x1 + main_box.w - 1;
	main_box.y2 = main_box.y1 + main_box.w - 1;

	box_gfx.w = main_box.w + 2;
	box_gfx.x1 = main_box.x1 - 1;
	box_gfx.y1 = main_box.y1 - 1;
	box_gfx.x2 = main_box.x2 + 1;
	box_gfx.y2 = main_box.y2 + 1;
	box_gfx.c1 = WHITE;
	box_gfx.c2 = BLACK;
	
	for (i = 0; i < NUM_SIZES; i++) {
		scale[i].w = 16;
		scale[i].x1 = 80 + i * 40;
		scale[i].y1 = 170;
		scale[i].x2 = scale[i].x1 + scale[i].w - 1;
		scale[i].y2 = scale[i].y1 + scale[i].w - 1;
		scale[i].c1 = WHITE;
		if (sprites[cur_sprite].size == 8 << i)
			scale[i].c2 = WHITE;
		else 
			scale[i].c2 = WHITE & CLEAR;
	}
		
	for (i = 0; i < NUM_COLORS; i++) {
		colors[i].w = 16;
		colors[i].x1 = 2;
		colors[i].y1 = 18 * i + 2;
		colors[i].x2 = colors[i].x1 + colors[i].w - 1;
		colors[i].y2 = colors[i].y1 + colors[i].w - 1;
		colors[i].c1 = WHITE;
	}
	colors[0].c2 = BLACK & CLEAR;
	colors[1].c2 = WHITE;
	colors[2].c2 = RED;
	colors[3].c2 = RGB15(31,15,0) | (1<<15);
	colors[4].c2 = RGB15(31,31,0) | (1<<15);
	colors[5].c2 = GREEN;
	colors[6].c2 = BLUE;
	colors[7].c2 = RGB15(15,0,31) | (1<<15);
	colors[8].c2 = RGB15(11,11,11) | (1<<15);
	colors[9].c2 = RGB15(22,22,22) | (1<<15);
	u16 color = colors[1].c2;
	
	cur_color.w = 32;
	cur_color.x1 = 20;
	cur_color.y1 = 2;
	cur_color.x2 = cur_color.x1 + cur_color.w - 1;
	cur_color.y2 = cur_color.y1 + cur_color.w - 1;
	cur_color.c1 = WHITE;
	cur_color.c2 = color;

	eraser.w = 16;
	eraser.x1 = 230;
	eraser.y1 = 16;
	eraser.x2 = eraser.x1 + eraser.w - 1;
	eraser.y2 = eraser.y1 + eraser.w - 1;
	eraser.c1 = WHITE;
	eraser.c2 = RED;
	
	reraser.w = 16;
	reraser.x1 = 230;
	reraser.y1 = 64;
	reraser.x2 = reraser.x1 + reraser.w - 1;
	reraser.y2 = reraser.y1 + reraser.w - 1;
	reraser.c1 = WHITE;
	reraser.c2 = GREEN;
	
	go.w = 32;
	go.x1 = 220;
	go.y1 = 150;
	go.x2 = go.x1 + go.w - 1;
	go.y2 = go.y1 + go.w - 1;
	go.c1 = WHITE;
	go.c2 = BLACK;
	
	draw_box(&box_gfx);
	for (i = 0; i < NUM_SIZES; i++)
		draw_box(&scale[i]);
	for (i = 0; i < NUM_COLORS; i++)
		draw_box(&colors[i]);
	draw_box(&cur_color);
	draw_box(&eraser);
	draw_box(&reraser);
	draw_box(&go);
	
	for (i = 0; i < sprites[cur_sprite].size * sprites[cur_sprite].size; i++) {
		dot.w = main_box.w / sprites[cur_sprite].size;
		dot.x1 = main_box.x1 + (i % sprites[cur_sprite].size) * (main_box.w / sprites[cur_sprite].size);
		dot.y1 = main_box.y1 + (i / sprites[cur_sprite].size) * (main_box.w / sprites[cur_sprite].size);
		dot.x2 = dot.x1 + dot.w - 1;
		dot.y2 = dot.y1 + dot.w - 1;
		dot.c1 = SPRITE_GFX[i + sprites[cur_sprite].gfx];
		dot.c2 = dot.c1;

		draw_box(&dot);
	}
	
	
	while(1) {
		swiWaitForVBlank();
		
		touch = touchReadXY();
		
		if (touch.px) {
			if (is_in(touch.px, touch.py, &main_box)) {
				dot.w = main_box.w / sprites[cur_sprite].size;
				dot.x1 = touch.px - touch.px % dot.w;
				dot.y1 = touch.py - touch.py % dot.w;
				dot.x2 = dot.x1 + dot.w - 1;
				dot.y2 = dot.y1 + dot.w - 1;
				dot.c1 = color;
				dot.c2 = color;
				
				SPRITE_GFX[(touch.px - main_box.x1) / (main_box.w / sprites[cur_sprite].size) + ((touch.py - main_box.y1) / (main_box.w / sprites[cur_sprite].size)) * sprites[cur_sprite].size + sprites[cur_sprite].gfx] = color; 

				draw_box(&dot);
			}
		}
		
		if (touch.px && !pressed) {
			for (i = 0; i < NUM_SIZES; i++) {
				if (is_in(touch.px, touch.py, &scale[i])) {
					for (j = 0; j < NUM_SIZES; j++)
						scale[j].c2 &= CLEAR;
						
					scale[i].c2 = WHITE;
					
					for (j = 0; j < NUM_SIZES; j++)
						draw_box(&scale[j]);
						
					sprites[cur_sprite].size = 8 << i;
					
					draw_box(&box_gfx);
					
					for (i = 0; i < sprites[cur_sprite].size * sprites[cur_sprite].size; i++) {
						dot.w = main_box.w / sprites[cur_sprite].size;
						dot.x1 = main_box.x1 + (i % sprites[cur_sprite].size) * (main_box.w / sprites[cur_sprite].size);
						dot.y1 = main_box.y1 + (i / sprites[cur_sprite].size) * (main_box.w / sprites[cur_sprite].size);
						dot.x2 = dot.x1 + dot.w - 1;
						dot.y2 = dot.y1 + dot.w - 1;
						dot.c1 = SPRITE_GFX[i + sprites[cur_sprite].gfx];
						dot.c2 = dot.c1;
						draw_box(&dot);
					}					
				}
			}
			
			for (i = 0; i < NUM_COLORS; i++) {
				if (is_in(touch.px, touch.py, &colors[i])) {
					color = colors[i].c2;
					cur_color.c2 = color;
					
					draw_box(&cur_color);
				}
			}
			
			if (is_in(touch.px, touch.py, &eraser)) {
				for (i = 0; i < sprites[cur_sprite].size * sprites[cur_sprite].size; i++)
					SPRITE_GFX[i + sprites[cur_sprite].gfx] = colors[0].c2;
				for (i = 0; i < sprites[cur_sprite].size * sprites[cur_sprite].size; i++) {
					dot.w = main_box.w / sprites[cur_sprite].size;
					dot.x1 = main_box.x1 + (i % sprites[cur_sprite].size) * (main_box.w / sprites[cur_sprite].size);
					dot.y1 = main_box.y1 + (i / sprites[cur_sprite].size) * (main_box.w / sprites[cur_sprite].size);
					dot.x2 = dot.x1 + dot.w - 1;
					dot.y2 = dot.y1 + dot.w - 1;
					dot.c1 = colors[0].c2;
					dot.c2 = colors[0].c2;
					
					draw_box(&dot);
				}
			}

			if (is_in(touch.px, touch.py, &reraser)) {
				for (i = 0; i < sprites[cur_sprite].size * sprites[cur_sprite].size; i++)
					SPRITE_GFX[i + sprites[cur_sprite].gfx] = sprite_buf[i];
				for (i = 0; i < sprites[cur_sprite].size * sprites[cur_sprite].size; i++) {
					dot.w = main_box.w / sprites[cur_sprite].size;
					dot.x1 = main_box.x1 + (i % sprites[cur_sprite].size) * (main_box.w / sprites[cur_sprite].size);
					dot.y1 = main_box.y1 + (i / sprites[cur_sprite].size) * (main_box.w / sprites[cur_sprite].size);
					dot.x2 = dot.x1 + dot.w - 1;
					dot.y2 = dot.y1 + dot.w - 1;
					dot.c1 = sprite_buf[i];
					dot.c2 = sprite_buf[i];
					
					draw_box(&dot);
				}
			}
			
			if (is_in(touch.px, touch.py, &go)) {
				return;
			}
		}
		
		if (touch.px)
			pressed = 1;
		else
			pressed = 0;
	}
}

void draw_box(edit_box* box) {
	int i;

	for (i = 0; i < box->w * box->w; i++)
		BG_GFX_SUB[(box->x1 + i % box->w) + (box->y1 + i / box->w) * SCREEN_WIDTH] = box->c2;
	
	for (i = 0; i < box->w; i++) {
		BG_GFX_SUB[box->x1 + i + box->y1 * SCREEN_WIDTH] = box->c1;
		BG_GFX_SUB[box->x1 + (box->y1 + i) * SCREEN_WIDTH] = box->c1;
		BG_GFX_SUB[box->x1 + i + box->y2 * SCREEN_WIDTH] = box->c1;
		BG_GFX_SUB[box->x2 + (box->y1 + i) * SCREEN_WIDTH] = box->c1;
	}
}

int is_in(int x, int y, edit_box* box) {
	if (x > box->x1 && x < box->x2 && y > box->y1 && y < box->y2)
		return 1;
	else
		return 0;
}

void show_sprite(int cur_sprite) {
	init_oam();
	init_screen();
	
	int i;
	int pressed = 1;
	int keys;
	int held;
	
	dmaCopy(show_screen_bin, BG_GFX_SUB, SCREEN_WIDTH * SCREEN_HEIGHT * 2);

	sprites[cur_sprite].x1 = SCREEN_WIDTH / 2 - sprites[cur_sprite].size;
	sprites[cur_sprite].y1 = SCREEN_HEIGHT / 2 - sprites[cur_sprite].size;
	
	sprites[cur_sprite].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE_DOUBLE | sprites[cur_sprite].y1;
	if (sprites[cur_sprite].size == 8) 
		sprites[cur_sprite].oam->attribute[1] = ATTR1_SIZE_8;
	if (sprites[cur_sprite].size == 16) 
		sprites[cur_sprite].oam->attribute[1] = ATTR1_SIZE_16;
	if (sprites[cur_sprite].size == 32) 
		sprites[cur_sprite].oam->attribute[1] = ATTR1_SIZE_32;
	sprites[cur_sprite].oam->attribute[1] |= sprites[cur_sprite].x1;
	sprites[cur_sprite].oam->attribute[2] = ATTR2_ALPHA(1) | (cur_sprite * 32);
	
	sprites[cur_sprite].rot->hdx=(1<<8);
	sprites[cur_sprite].rot->hdy=0;
	sprites[cur_sprite].rot->vdx=0;
	sprites[cur_sprite].rot->vdy=(1<<8);
		
	update_oam();

	edit_box rot_box[2];
	edit_box exit_box;

	exit_box.w = 32;
	exit_box.x1 = 220;
	exit_box.y1 = 150;
	exit_box.x2 = exit_box.x1 + exit_box.w - 1;
	exit_box.y2 = exit_box.y1 + exit_box.w - 1;
	exit_box.c1 = WHITE;
	exit_box.c2 = BLACK;
	draw_box(&exit_box);

	for (i = 0; i < 2; i++) {
		rot_box[i].w = 32;
		rot_box[i].x1 = SCREEN_WIDTH / 2 - 40 + i * 56;
		rot_box[i].y1 = SCREEN_HEIGHT / 2 - 16;
		rot_box[i].x2 = rot_box[i].x1 + rot_box[i].w - 1;
		rot_box[i].y2 = rot_box[i].y1 + rot_box[i].w - 1;
		rot_box[i].c1 = WHITE;
		rot_box[i].c2 = BLACK;
		draw_box(&rot_box[i]);
	}
	
	int angle = 0;

	while (1) {
		swiWaitForVBlank();
		
		touch = touchReadXY();
		scanKeys();
		keys = keysDown();
		held = keysHeld();
		
		if (touch.px || (held & KEY_R) || (held & KEY_L)) {
			if (is_in(touch.px, touch.py, &rot_box[1]) || (held & KEY_R)) {
				angle = angle + 2;
				sprites[cur_sprite].rot->hdx = COS[angle & 0x1FF] >> 4;
				sprites[cur_sprite].rot->hdy = SIN[angle & 0x1FF] >> 4;
				sprites[cur_sprite].rot->vdx = -sprites[cur_sprite].rot->hdy;
				sprites[cur_sprite].rot->vdy = sprites[cur_sprite].rot->hdx;
			}
			
			if (is_in(touch.px, touch.py, &rot_box[0]) || (held & KEY_L)) {
				angle = angle - 2;
				sprites[cur_sprite].rot->hdx = COS[angle & 0x1FF] >> 4;
				sprites[cur_sprite].rot->hdy = SIN[angle & 0x1FF] >> 4;
				sprites[cur_sprite].rot->vdx = -sprites[cur_sprite].rot->hdy;
				sprites[cur_sprite].rot->vdy = sprites[cur_sprite].rot->hdx;
			}
		}
		
		if (keys & KEY_START)
			return;

		if (touch.px && !pressed) {
			if (is_in(touch.px, touch.py, &exit_box))
				return;
		}
			
		if (touch.px)
			pressed = 1;
		else
			pressed = 0;
		
		update_oam();
	}
}

int load_sprite(int cur_sprite) {
	init_oam();
	init_screen();
	
	dmaCopy(load_screen_bin, BG_GFX, SCREEN_WIDTH * SCREEN_HEIGHT * 2);

	
	int i, j;
	int pressed = 1;
	
	edit_box option[NUM_USER_SPRITES];
	
	for (i = 0; i < NUM_USER_SPRITES; i++) {
		option[i].w = sprites[i].size + 2;
		option[i].x1 = i * 35;
		option[i].y1 = 0;
		option[i].x2 = option[i].x1 + option[i].w - 1;
		option[i].y2 = option[i].y1 + option[i].w - 1;
		option[i].c1 = WHITE;
		option[i].c2 = BLACK;
		draw_box(&option[i]);
	}
	
	edit_box exit_box;
	
	exit_box.w = 32;
	exit_box.x1 = 220;
	exit_box.y1 = 150;
	exit_box.x2 = exit_box.x1 + exit_box.w - 1;
	exit_box.y2 = exit_box.y1 + exit_box.w - 1;
	exit_box.c1 = WHITE;
	exit_box.c2 = BLACK;
	draw_box(&exit_box);
	
	for (i = 0; i < NUM_USER_SPRITES; i++)
		for (j = 0; j < sprites[i].size * sprites[i].size; j++)
			BG_GFX_SUB[j % sprites[i].size + (j / sprites[i].size) * SCREEN_WIDTH + i * 35 + 1 + SCREEN_WIDTH] = SPRITE_GFX[j + sprites[i].gfx];

	while (1) {
		touch = touchReadXY();
	
		if (touch.px && !pressed) {
			for (i = 0; i < NUM_USER_SPRITES; i++) {
				if (is_in(touch.px, touch.py, &option[i])) {
					return i;
				}
			}
			
			if (is_in(touch.px, touch.py, &exit_box))
				return cur_sprite;
		}
		
		if (touch.px)
			pressed = 1;
		else
			pressed = 0;

	}
}

void play_game(void) {
	init_oam();
	init_screen();
	dmaCopy(SPRITE_GFX, SPRITE_GFX_SUB, 32*32*2*NUM_SPRITES);

	int NUM_GUNS = 5;
	//int NUM_BGUNS = 3;
	int NUM_BAD = 9;

	int i, j;

	shiny_star stars[NUM_STARS];
	
	for (i = 0; i < NUM_STARS; i++) {
		stars[i].x = rand() % SCREEN_WIDTH;
		stars[i].y = rand() % SCREEN_HEIGHT;
		stars[i].s = rand() % 2;
		stars[i].c = rand() | RGB15(28, 28, 31) | BIT(15);
	}
		
	sprite ship;
	sprite bad[NUM_BAD];
	sprite guns[NUM_GUNS];
	
	ship = sprites[0];
	ship.id = 0;
	ship.a = 1;
	
	ship.x1 = SCREEN_WIDTH / 2 - ship.size;
	ship.y1 = SCREEN_HEIGHT - ship.size - 5;
	ship.dx = 0;
	ship.dy = 0;
	ship.s = 0;
	ship.oam = &oam_copy[ship.id];
	ship.rot = &sprite_rotation[ship.id];
	
	ship.oam->attribute[0] = ATTR0_DISABLED;
	if (ship.size == 8) 
		ship.oam->attribute[1] = ATTR1_SIZE_8;
	else if (ship.size == 16) 
		ship.oam->attribute[1] = ATTR1_SIZE_16;
	else if (ship.size == 32) 
		ship.oam->attribute[1] = ATTR1_SIZE_32;
	else
		return;

	ship.oam->attribute[1] |= ship.x1;
	ship.oam->attribute[2] = ATTR2_ALPHA(1);
		
	ship.rot->hdx=(1<<8);
	ship.rot->hdy=0;
	ship.rot->vdx=0;
	ship.rot->vdy=(1<<8);
	
	for (i = 0; i < ship.size * ship.size ; i++) {
		if ((SPRITE_GFX[i + ship.gfx] & (1<<15)) != 0) {
			ship.fx = i % ship.size;
			ship.fy = i / ship.size;
			break;
		}
	}

	for (i = 0; i < NUM_BAD; i++) {
		int eh = rand() % 3;
		bad[i] = sprites[1 + eh];
		bad[i].id = i+1;
		bad[i].a = 1;
		bad[i].x1 = rand() % (SCREEN_WIDTH - bad[i].size);
		bad[i].y1 = rand() % (SCREEN_HEIGHT - bad[i].size);
		bad[i].dx = rand() % 3 + 1;
		if (rand() % 2) bad[i].dx = -bad[i].dx;
		bad[i].dy = rand() % 3 + 1;
		bad[i].s = 1;
		bad[i].oam = &oam_copy[bad[i].id];
		bad[i].oam->attribute[2] = ATTR2_ALPHA(1) | (32 + (eh * 32));
		bad[i].rot = &sprite_rotation[bad[i].id];
		
		bad[i].oam->attribute[0] = ATTR0_DISABLED;
		if (bad[i].size == 8)
			bad[i].oam->attribute[1] = ATTR1_SIZE_8;
		else if (bad[i].size == 16) 
			bad[i].oam->attribute[1] = ATTR1_SIZE_16;
		else if (bad[i].size == 32)
			bad[i].oam->attribute[1] = ATTR1_SIZE_32;
		else
			return;
	
		bad[i].oam->attribute[1] |= bad[i].x1;
		
		bad[i].rot->hdx=(1<<8);
		bad[i].rot->hdy=0;
		bad[i].rot->vdx=0;
		bad[i].rot->vdy=(1<<8);
	}

	for (i = 0; i < NUM_GUNS; i++) {
		guns[i] = sprites[4];
		guns[i].id = i+NUM_BAD+1;
		guns[i].a = 0;
		guns[i].x1 = 0;
		guns[i].y1 = 0;
		guns[i].dx = 0;
		guns[i].dy = 0;
		guns[i].s = 0;
		guns[i].oam = &oam_copy[guns[i].id];
		guns[i].oam->attribute[2] = ATTR2_ALPHA(1) | (32 * 4);
		guns[i].rot = &sprite_rotation[guns[i].id];
		
		guns[i].oam->attribute[0] = ATTR0_DISABLED;
		if (guns[i].size == 8)
			guns[i].oam->attribute[1] = ATTR1_SIZE_8;
		else if (guns[i].size == 16) 
			guns[i].oam->attribute[1] = ATTR1_SIZE_16;
		else if (guns[i].size == 32)
			guns[i].oam->attribute[1] = ATTR1_SIZE_32;
		else
			return;
			
		guns[i].oam->attribute[1] |= guns[i].x1;
		
		guns[i].rot->hdx=(1<<8);
		guns[i].rot->hdy=0;
		guns[i].rot->vdx=0;
		guns[i].rot->vdy=(1<<8);

		for (j = 0; j < guns[i].size * guns[i].size ; j++) {
			if ((SPRITE_GFX[j + guns[i].gfx] & (1<<15)) != 0) {
				guns[i].fx = j % guns[i].size;
				guns[i].fy = j / guns[i].size;
				break;
			}
		}
	}


	for (i = 0; i < 128; i++)
		oam_copy_sub[i] = oam_copy[i];
	
	ship.oam = &oam_copy_sub[ship.id];
	ship.rot = &sprite_rotation_sub[ship.id];
	ship.oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE | ship.y1;
	
	for (i = 0; i < NUM_BAD; i++)
		bad[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE | bad[i].y1;
		
	for (i = 0; i < NUM_GUNS; i++) {
		guns[i].oam = &oam_copy_sub[guns[i].id];
		guns[i].rot = &sprite_rotation_sub[guns[i].id];
	}
	
	update_oam();

	int pressed = 1;
	int keys;
	int held;
	int GOD = 1;
	
	while(1) {		
		touch = touchReadXY();
		scanKeys();
		keys = keysDown();
		held = keysHeld();

		if (held & KEY_LEFT)
			ship.dx = -2;
		else if (held & KEY_RIGHT)
			ship.dx = 2;
		else
			ship.dx = 0;
		
		if (keys & KEY_R)
			GOD = -GOD;
		
		if (held & KEY_UP)
			ship.dy = -2;
		else if (held & KEY_DOWN)
			ship.dy = 2;
		else
			ship.dy = 0;

		ship.x1 += ship.dx;
		ship.y1 += ship.dy;
		
		if (ship.x1 < 0) ship.x1 = 0;
		if (ship.x1 > SCREEN_WIDTH - ship.size) ship.x1 = SCREEN_WIDTH - ship.size;
		if (ship.y1 < 0) ship.y1 = 0;
		if (ship.y1 > SCREEN_HEIGHT - ship.size) ship.y1 = SCREEN_HEIGHT - ship.size;
		
		if (keys & KEY_A && ship.a) {
			for (i = 0; i < NUM_GUNS; i++) {
				if (guns[i].a == 0) {
					guns[i].a = 1;
					guns[i].s = 0;
					guns[i].x1 = ship.x1 + ship.fx - guns[i].fx;
					if (guns[i].x1 < 0) { guns[i].x1 = 0; }
					guns[i].y1 = ship.y1 + ship.fy - guns[i].fy;
					guns[i].dy = -3;
					guns[i].oam = &oam_copy_sub[guns[i].id];
					guns[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
					break;
				}
			}
		}

		for (i = 0; i < NUM_BAD; i++) {
			if (bad[i].a) {
				bad[i].x1 += bad[i].dx;
				bad[i].y1 += bad[i].dy;
				
				if (bad[i].y1 > SCREEN_HEIGHT) {
					bad[i].y1 = 0;
					if (bad[i].s) {
						bad[i].oam->attribute[0] = ATTR0_DISABLED;
						bad[i].oam = &oam_copy_sub[bad[i].id];
						bad[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
						bad[i].s = 0;
					} else {
						bad[i].oam->attribute[0] = ATTR0_DISABLED;
						bad[i].oam = &oam_copy[bad[i].id];
						bad[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
						bad[i].s = 1;
						bad[i].x1 = rand() % (SCREEN_WIDTH - bad[i].size);
						bad[i].dx = rand() % 3 + 1;
						if (rand() % 2) bad[i].dx = -bad[i].dx;
						bad[i].dy = rand() % 3 + 1;
					}
				}
				
				if (bad[i].x1 < 0 || bad[i].x1 > SCREEN_WIDTH - bad[i].size) {
					bad[i].dx = -bad[i].dx;
					bad[i].x1 += bad[i].dx;
				}
			} else {
				bad[i].oam->attribute[0] = ATTR0_DISABLED;
				bad[i].oam = &oam_copy[bad[i].id];
				bad[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
				bad[i].s = 1;
				bad[i].x1 = rand() % (SCREEN_WIDTH - bad[i].size);
				bad[i].y1 = 0;
				bad[i].dx = rand() % 3 + 1;
				if (rand() % 2) bad[i].dx = -bad[i].dx;
				bad[i].dy = rand() % 3 + 1;
				bad[i].a = 1;
			}
		}

		for (i = 0; i < NUM_GUNS; i++) {
			if (guns[i].a) {
				guns[i].x1 += guns[i].dx;
				guns[i].y1 += guns[i].dy;
				
				if (guns[i].y1 < 0) {
					if (guns[i].s) {
						guns[i].oam->attribute[0] = ATTR0_DISABLED;
						guns[i].oam = &oam_copy_sub[guns[i].id];
						guns[i].s = 0;
						guns[i].a = 0;
					} else {
						guns[i].y1 = SCREEN_HEIGHT + guns[i].fy;
						guns[i].oam->attribute[0] = ATTR0_DISABLED;
						guns[i].oam = &oam_copy[guns[i].id];
						guns[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
						guns[i].s = 1;
					}
				}
			}
		}
		
		if ((touch.px && !pressed) || (keys & KEY_START))
			return;
			
		if (touch.px)
			pressed = 1;
		else
			pressed = 0;
		
		swiWaitForVBlank();
		
		for (i = 0; i < NUM_STARS; i++) {
			if (stars[i].s) {
				BG_GFX[stars[i].x + stars[i].y * SCREEN_WIDTH] = BLACK;
				stars[i].y += 3;
				if (stars[i].y > SCREEN_HEIGHT) {
					stars[i].s = 0;
					stars[i].y = 0;
				} else {
					BG_GFX[stars[i].x + stars[i].y * SCREEN_WIDTH] = stars[i].c;
				}
			} else {
				BG_GFX_SUB[stars[i].x + stars[i].y * SCREEN_WIDTH] = BLACK;
				stars[i].y += 3;
				if (stars[i].y > SCREEN_HEIGHT) {
					stars[i].s = 1;
					stars[i].y = 0;
					stars[i].x = rand() % SCREEN_WIDTH;
				} else {
					BG_GFX_SUB[stars[i].x + stars[i].y * SCREEN_WIDTH] = stars[i].c;
				}
			}
		}

		ship.oam->attribute[1] &= 0xFE00;
		ship.oam->attribute[1] |= ship.x1;
			
		ship.oam->attribute[0] &= 0xFF00;
		ship.oam->attribute[0] |= ship.y1;
		
		for (i = 0; i < NUM_BAD; i++) {
			if (bad[i].a) {
				bad[i].oam->attribute[1] &= 0xFE00;
				bad[i].oam->attribute[1] |= bad[i].x1;
				
				bad[i].oam->attribute[0] &= 0xFF00;
				bad[i].oam->attribute[0] |= bad[i].y1;
				
				if (ship.a && (GOD != -1)) {
					if (collide(&ship, &bad[i])) {
						ship.oam->attribute[0] = ATTR0_DISABLED;
						ship.a = 0;
					}
				}
			}
		}

		for (i = 0; i < NUM_GUNS; i++) {
			if (guns[i].a) {
				guns[i].oam->attribute[1] &= 0xFE00;
				guns[i].oam->attribute[1] |= guns[i].x1;
				
				guns[i].oam->attribute[0] &= 0xFF00;
				guns[i].oam->attribute[0] |= guns[i].y1;
				
				for (j = 0; j < NUM_BAD; j++) {
					if (bad[j].a) {
						if (collide(&guns[i], &bad[j])) {
							bad[j].oam->attribute[0] = ATTR0_DISABLED;
							bad[j].a = 0;
							guns[i].oam->attribute[0] = ATTR0_DISABLED;
							guns[i].a = 0;
						}
					}
				}
			}
		}
		
		for (i = 0; i < NUM_GUNS; i++) {
			if (guns[i].a) {
				guns[i].oam->attribute[1] &= 0xFE00;
				guns[i].oam->attribute[1] |= guns[i].x1;
				
				guns[i].oam->attribute[0] &= 0xFF00;
				guns[i].oam->attribute[0] |= guns[i].y1;
			}
		}

		update_oam();
	}
}

int collide(sprite* one, sprite* two) {
	u16* gfx = (one->s) ? (u16*)SPRITE_GFX : (u16*)SPRITE_GFX_SUB;
	one->x2 = one->x1 + one->size - 1;
	one->y2 = one->y1 + one->size - 1;
	two->x2 = two->x1 + two->size - 1;
	two->y2 = two->y1 + two->size - 1;

	if (one->s == two->s) {
       if ((one->x2 > two->x1) && (one->x1 < two->x2) && (one->y2 > two->y1) && (one->y1 < two->y2)) {
			int x1 = (one->x1 > two->x1) ? one->x1 : two->x1;
			int x2 = (one->x1 + one->size < two->x1 + two->size) ? one->x1 + one->size : two->x1 + two->size;
			int y1 = (one->y1 > two->y1) ? one->y1 : two->y1;
			int y2 = (one->y1 + one->size < two->y1 + two->size) ? one->y1 + one->size: two->y1 + two->size;

			int x_one = (one->x1 > x1) ? 0 : x1 - one->x1;
			int y_one = (one->y1 > y1) ? 0 : y1 - one->y1;
			int x_two = (two->x1 > x1) ? 0 : x1 - two->x1;
			int y_two = (two->y1 > y1) ? 0 : y1 - two->y1;
						
			int x, y;
			for (y = 0; y < y2 - y1; y++) {
				for (x = 0; x < x2 - x1; x++) {
					if (gfx[one->gfx + x_one + y_one * one->size + x + y * one->size] & (1<<15)) {
						if (gfx[two->gfx + x_two + y_two * one->size + x + y * two->size] & (1<<15)) {
							return 1;
						}
					}
				}
			}
		}
	}
	
	return 0;
}

void play_game2() {
	init_oam();
	init_screen();
	dmaCopy(SPRITE_GFX, SPRITE_GFX_SUB, 32*32*2*NUM_SPRITES);

	int NUM_GUNS = 5;

	int i, j;

	sprite ship;
	sprite ship2;
	sprite guns[NUM_GUNS];
	sprite guns2[NUM_GUNS];
	
	ship = sprites[0];
	ship.id = 0;
	ship.a = 1;
	
	ship.x1 = SCREEN_WIDTH / 2 - ship.size;
	ship.y1 = SCREEN_HEIGHT - ship.size - 5;
	ship.dx = 0;
	ship.dy = 0;
	ship.s = 0;
	ship.oam = &oam_copy[ship.id];
	ship.rot = &sprite_rotation[ship.id];
	
	ship.oam->attribute[0] = ATTR0_DISABLED;
	if (ship.size == 8) 
		ship.oam->attribute[1] = ATTR1_SIZE_8;
	else if (ship.size == 16) 
		ship.oam->attribute[1] = ATTR1_SIZE_16;
	else if (ship.size == 32) 
		ship.oam->attribute[1] = ATTR1_SIZE_32;
	else
		return;

	ship.oam->attribute[1] |= ship.x1;
	ship.oam->attribute[2] = ATTR2_ALPHA(1);
		
	ship.rot->hdx=(1<<8);
	ship.rot->hdy=0;
	ship.rot->vdx=0;
	ship.rot->vdy=(1<<8);
	
	for (i = 0; i < ship.size * ship.size ; i++) {
		if ((SPRITE_GFX[i + ship.gfx] & (1<<15)) != 0) {
			ship.fx = i % ship.size;
			ship.fy = i / ship.size;
			break;
		}
	}

	ship2 = sprites[5];
	ship2.id = 1;
	ship2.a = 1;
	
	ship2.x1 = SCREEN_WIDTH / 2 - ship2.size;
	ship2.y1 = 0;
	ship2.dx = 0;
	ship2.dy = 0;
	ship2.s = 1;
	ship2.oam = &oam_copy[ship2.id];
	ship2.rot = &sprite_rotation[ship2.id];
	
	ship2.oam->attribute[0] = ATTR0_DISABLED;
	if (ship2.size == 8) 
		ship2.oam->attribute[1] = ATTR1_SIZE_8;
	else if (ship2.size == 16) 
		ship2.oam->attribute[1] = ATTR1_SIZE_16;
	else if (ship2.size == 32) 
		ship2.oam->attribute[1] = ATTR1_SIZE_32;
	else
		return;

	ship2.oam->attribute[1] |= ship2.x1;
	ship2.oam->attribute[2] = ATTR2_ALPHA(1) | (32 * 5);
		
	ship2.rot->hdx=(1<<8);
	ship2.rot->hdy=0;
	ship2.rot->vdx=0;
	ship2.rot->vdy=(1<<8);
	
	for (i = ship2.size * ship2.size; i > 0; i--) {
		if ((SPRITE_GFX[i + ship2.gfx] & (1<<15)) != 0) {
			ship2.fx = i % ship2.size;
			ship2.fy = i / ship2.size;
			break;
		}
	}


	for (i = 0; i < NUM_GUNS; i++) {
		guns[i] = sprites[4];
		guns[i].id = i+2;
		guns[i].a = 0;
		guns[i].x1 = 0;
		guns[i].y1 = 0;
		guns[i].dx = 0;
		guns[i].dy = 0;
		guns[i].s = 0;
		guns[i].oam = &oam_copy[guns[i].id];
		guns[i].oam->attribute[2] = ATTR2_ALPHA(1) | (32 * 4);
		guns[i].rot = &sprite_rotation[guns[i].id];
		
		guns[i].oam->attribute[0] = ATTR0_DISABLED;
		if (guns[i].size == 8)
			guns[i].oam->attribute[1] = ATTR1_SIZE_8;
		else if (guns[i].size == 16) 
			guns[i].oam->attribute[1] = ATTR1_SIZE_16;
		else if (guns[i].size == 32)
			guns[i].oam->attribute[1] = ATTR1_SIZE_32;
		else
			return;
			
		guns[i].oam->attribute[1] |= guns[i].x1;
		
		guns[i].rot->hdx=(1<<8);
		guns[i].rot->hdy=0;
		guns[i].rot->vdx=0;
		guns[i].rot->vdy=(1<<8);

		for (j = 0; j < guns[i].size * guns[i].size ; j++) {
			if ((SPRITE_GFX[j + guns[i].gfx] & (1<<15)) != 0) {
				guns[i].fx = j % guns[i].size;
				guns[i].fy = j / guns[i].size;
				break;
			}
		}

		guns2[i] = sprites[6];
		guns2[i].id = i+NUM_GUNS+2;
		guns2[i].a = 0;
		guns2[i].x1 = 0;
		guns2[i].y1 = 0;
		guns2[i].dx = 0;
		guns2[i].dy = 0;
		guns2[i].s = 1;
		guns2[i].oam = &oam_copy[guns2[i].id];
		guns2[i].oam->attribute[2] = ATTR2_ALPHA(1) | (32 * 6);
		guns2[i].rot = &sprite_rotation[guns2[i].id];
		
		guns2[i].oam->attribute[0] = ATTR0_DISABLED;
		if (guns2[i].size == 8)
			guns2[i].oam->attribute[1] = ATTR1_SIZE_8;
		else if (guns2[i].size == 16) 
			guns2[i].oam->attribute[1] = ATTR1_SIZE_16;
		else if (guns2[i].size == 32)
			guns2[i].oam->attribute[1] = ATTR1_SIZE_32;
		else
			return;
			
		guns2[i].oam->attribute[1] |= guns2[i].x1;
		
		guns2[i].rot->hdx=(1<<8);
		guns2[i].rot->hdy=0;
		guns2[i].rot->vdx=0;
		guns2[i].rot->vdy=(1<<8);

		for (j = guns2[i].size * guns2[i].size; j > 0; j--) {
			if ((SPRITE_GFX[j + guns2[i].gfx] & (1<<15)) != 0) {
				guns2[i].fx = j % guns2[i].size;
				guns2[i].fy = j / guns2[i].size;
				break;
			}
		}
	}


	for (i = 0; i < 128; i++)
		oam_copy_sub[i] = oam_copy[i];
	
	ship.oam = &oam_copy_sub[ship.id];
	ship.rot = &sprite_rotation_sub[ship.id];
	ship.oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE | ship.y1;

	ship2.oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE | ship2.y1;
	
	for (i = 0; i < NUM_GUNS; i++) {
		guns[i].oam = &oam_copy_sub[guns[i].id];
		guns[i].rot = &sprite_rotation_sub[guns[i].id];
	}
	
	update_oam();

	int pressed = 1;
	int keys;
	int held;
	
	while(1) {		
		touch = touchReadXY();
		scanKeys();
		keys = keysDown();
		held = keysHeld();

		if (held & KEY_LEFT)
			ship.dx = -2;
		else if (held & KEY_RIGHT)
			ship.dx = 2;
		else
			ship.dx = 0;
		
		if (held & KEY_UP)
			ship.dy = -2;
		else if (held & KEY_DOWN)
			ship.dy = 2;
		else
			ship.dy = 0;

		//
		if (held & KEY_Y)
			ship2.dx = -2;
		else if (held & KEY_A)
			ship2.dx = 2;
		else
			ship2.dx = 0;
		
		if (held & KEY_X)
			ship2.dy = -2;
		else if (held & KEY_B)
			ship2.dy = 2;
		else
			ship2.dy = 0;

		ship.x1 += ship.dx;
		ship.y1 += ship.dy;
		
		ship2.x1 += ship2.dx;
		ship2.y1 += ship2.dy;

		if (ship.x1 < 0) ship.x1 = 0;
		if (ship.x1 > SCREEN_WIDTH - ship.size) ship.x1 = SCREEN_WIDTH - ship.size;
		if (ship.y1 < 0) ship.y1 = 0;
		if (ship.y1 > SCREEN_HEIGHT - ship.size) ship.y1 = SCREEN_HEIGHT - ship.size;

		if (ship2.x1 < 0) ship2.x1 = 0;
		if (ship2.x1 > SCREEN_WIDTH - ship2.size) ship2.x1 = SCREEN_WIDTH - ship2.size;
		if (ship2.y1 < 0) ship2.y1 = 0;
		if (ship2.y1 > SCREEN_HEIGHT - ship2.size) ship2.y1 = SCREEN_HEIGHT - ship2.size;
		
		if (keys & KEY_L && ship.a) {
			for (i = 0; i < NUM_GUNS; i++) {
				if (guns[i].a == 0) {
					guns[i].a = 1;
					guns[i].s = 0;
					guns[i].x1 = ship.x1 + ship.fx - guns[i].fx;
					guns[i].y1 = ship.y1 + ship.fy - guns[i].fy;
					guns[i].dy = -3;
					guns[i].oam = &oam_copy_sub[guns[i].id];
					guns[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
					break;
				}
			}
		}

		if (keys & KEY_R && ship2.a) {
			for (i = 0; i < NUM_GUNS; i++) {
				if (guns2[i].a == 0) {
					guns2[i].a = 1;
					guns2[i].s = 1;
					guns2[i].x1 = ship2.x1 + ship2.fx - guns2[i].fx;
					guns2[i].y1 = ship2.y1 + ship2.fy - guns2[i].fy;
					guns2[i].dy = 3;
					guns2[i].oam = &oam_copy[guns2[i].id];
					guns2[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
					break;
				}
			}
		}

		for (i = 0; i < NUM_GUNS; i++) {
			if (guns[i].a) {
				guns[i].x1 += guns[i].dx;
				guns[i].y1 += guns[i].dy;
				
				if (guns[i].y1 < 0) {
					if (guns[i].s) {
						guns[i].oam->attribute[0] = ATTR0_DISABLED;
						guns[i].oam = &oam_copy_sub[guns[i].id];
						guns[i].s = 0;
						guns[i].a = 0;
					} else {
						guns[i].y1 = SCREEN_HEIGHT + guns[i].fy;
						guns[i].oam->attribute[0] = ATTR0_DISABLED;
						guns[i].oam = &oam_copy[guns[i].id];
						guns[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
						guns[i].s = 1;
					}
				}
			}

			if (guns2[i].a) {
				guns2[i].x1 += guns2[i].dx;
				guns2[i].y1 += guns2[i].dy;
				
				if (guns2[i].y1 > SCREEN_HEIGHT - guns2[i].fy) {
					if (guns2[i].s) {
						guns2[i].y1 = guns2[i].size - guns2[i].fy;
						guns2[i].oam->attribute[0] = ATTR0_DISABLED;
						guns2[i].oam = &oam_copy_sub[guns2[i].id];
						guns2[i].oam->attribute[0] = ATTR0_BMP | ATTR0_ROTSCALE;
						guns2[i].s = 0;
					} else {
						guns2[i].oam->attribute[0] = ATTR0_DISABLED;
						guns2[i].s = 1;
						guns2[i].a = 0;
					}
				}
			}
		}
		
		if ((touch.px && !pressed) || (keys & KEY_START))
			return;
			
		if (touch.px)
			pressed = 1;
		else
			pressed = 0;
		
		swiWaitForVBlank();
		

		ship.oam->attribute[1] &= 0xFE00;
		ship.oam->attribute[1] |= ship.x1;
			
		ship.oam->attribute[0] &= 0xFF00;
		ship.oam->attribute[0] |= ship.y1;


		ship2.oam->attribute[1] &= 0xFE00;
		ship2.oam->attribute[1] |= ship2.x1;
			
		ship2.oam->attribute[0] &= 0xFF00;
		ship2.oam->attribute[0] |= ship2.y1;

		for (i = 0; i < NUM_GUNS; i++) {
			if (guns[i].a) {
				guns[i].oam->attribute[1] &= 0xFE00;
				guns[i].oam->attribute[1] |= guns[i].x1;
				
				guns[i].oam->attribute[0] &= 0xFF00;
				guns[i].oam->attribute[0] |= guns[i].y1;
				
				if (ship2.a && collide(&guns[i], &ship2)) {
					ship2.oam->attribute[0] = ATTR0_DISABLED;
					ship2.a--;
					guns[i].oam->attribute[0] = ATTR0_DISABLED;
					guns[i].a = 0;
				}
			}

			if (guns2[i].a) {
				guns2[i].oam->attribute[1] &= 0xFE00;
				guns2[i].oam->attribute[1] |= guns2[i].x1;
				
				guns2[i].oam->attribute[0] &= 0xFF00;
				guns2[i].oam->attribute[0] |= guns2[i].y1;
				
				if (ship.a && collide(&guns2[i], &ship)) {
					ship.oam->attribute[0] = ATTR0_DISABLED;
					ship.a--;
					guns2[i].oam->attribute[0] = ATTR0_DISABLED;
					guns2[i].a = 0;
				}
			}
		}
		
		update_oam();
	}
}
