#!/usr/bin/env python3

"""
#====================================================================
Module for the PixelImage class
~Graham
#====================================================================
"""

#====================================================================
import math
import random
import functools
import tkinter as tk

from PIL import Image, ImageDraw, ImageTk
#====================================================================

"""
#====================================================================
Some colors and methods for convenience
#====================================================================
"""
WHITE = (255, 255, 255, 255)
RED = (255, 0, 0, 255)
GREEN = (0, 255, 0, 255)
BLUE = (0, 0, 255, 255)
BLACK = (0, 0, 0, 255)

def ColorToHex(COLOR):
    (R, G, B, A) = COLOR
    return '#{:02X}{:02X}{:02X}'.format(R, G, B)

def InvertColor(COLOR):
    (R, G, B, A) = COLOR
    R = 255 - R
    G = 255 - G
    B = 255 - B
    return (R, G, B, A)

def ColorSum(COLOR):
    (R, G, B) = COLOR
    return (R + G + B)

"""
#================================================================
Generate a random colors to see sort in action.
#================================================================
"""
def RandomColor(fillType):
    (R, G, B, A) = (0, 0, 0, 255)

    #Pick an indiviual Hue
    if fillType == 1:
        rgb = random.choice('rgb')
        randomColor = random.randint(0, 255)
        if rgb == 'r':
            R = randomColor
        if rgb == 'g':
            G = randomColor
        if rgb == 'b':
            B = randomColor

    #Any random color will do
    if fillType == 2:
        R = random.randint(0, 255)
        G = random.randint(0, 255)
        B = random.randint(0, 255)

    return (R, G, B, A)

"""
#====================================================================
Class that adds some per-pixel functionality to the vanilla Tk
canvas, including a visual quicksort demo.

It's a lot more complex and slow to access data for individual pixels
in a Tk Canvas object, so we cheat and do most of our data
manipulation in a PIL Image object.
#====================================================================
"""
class PixelCanvas(tk.Canvas):
    def __init__(
            self,
            master,
            UpdateMethod,
            size = (100, 100),
            background = WHITE):
        tk.Canvas.__init__(self, master)
        self.master = master
        self.UpdateMethod = UpdateMethod
        self.config(
            width = size[0],
            height = size[1],
            background = ColorToHex(background))
        
        #An Image object to handle pixel data
        self.imageData = Image.new(
            "RGB",
            size,
            background)        
        self.draw = ImageDraw.Draw(self.imageData, "RGBA")

    """
    #================================================================
    Draw an oval in our Image object using our Draw object.
    #================================================================
    """
    def DrawOval(self, x, y, color = WHITE, size = 1):
        x1 = x - math.ceil(size / 2)
        y1 = y - math.ceil(size / 2)
        x2 = x + math.floor(size / 2)
        y2 = y + math.floor(size / 2)

        self.draw.ellipse(
            (x1, y1, x2, y2),
            fill = color,
            outline = color)
        
        self.Refresh()
    
    """
    #================================================================
    Since Tk Canvas objects don't have a great way to get per pixel
    data from them, we cheat and do all our drawing operations to a
    PIL Image object. This function copies that Image data back to
    the canvas.
    #================================================================
    """
    def Refresh(self):
        self.imageDataTk = ImageTk.PhotoImage(self.imageData)
        self.delete('all')
        self.create_image(
            0, 0, anchor = "nw", image = self.imageDataTk)
        self.UpdateMethod()

    """
    #================================================================
    """
    def GetPixelColor(self, x, y):
        return (*self.imageData.getpixel((x, y)), 255)

    """
    #================================================================
    Fill an empty space on this canvas with faux pixels.  This isn't
    really necessary, but it was useful for bug testing.
    #================================================================
    """
    def FillEmptyPixels(self, fillType):
        width = self.winfo_width() - 2
        height = self.winfo_height() - 2
        for y in range(height):
            for x in range(width):
                if self.GetPixelColor(x, y) == WHITE:
                    self.draw.point((x, y), RandomColor(fillType))
            self.Refresh()

    """
    #================================================================
    Quicksort a list of colors, displaying results live. Optimized
    for large stretches of repeated data.
    #================================================================
    """
    def DoVisualQuickSort(self, sortFlag):
        self.colorData = list(self.imageData.getdata())
        
        if sortFlag == 1:
            self.VisualQuickSortByHue(0, len(self.colorData) - 1)
            return
        
        self.VisualQuickSortByBrightness(0, len(self.colorData) - 1)

    """
    #================================================================
    This version sorts first by red, then green, then blue, and works
    well with the single hue random fill mode.
    #================================================================
    """
    def VisualQuickSortByHue(self, indexStart, indexEnd):
        colors = self.colorData
        if indexEnd <= indexStart:
            return

        #Optimized quicksort doesn't play nice if there are only
        #two pixels to swap, so we treat this as a special case.
        if indexEnd - indexStart == 1:
            if colors[indexStart] <= colors[indexEnd]:
                return
            self.SwapPixels(indexStart, indexEnd)
            return

        pivotStart, pivotEnd = self.PartitionColorsByHue(
            indexStart, indexEnd)

        #Sort everything that wasn't part of this set of pivots.
        self.VisualQuickSortByHue(
            indexStart, pivotStart - 1)
        self.VisualQuickSortByHue(
            pivotEnd, indexEnd)

    """
    #================================================================
    This version sorts by the sum R+G+B, and works well with the any
    color random fill mode.
    #================================================================
    """
    def VisualQuickSortByBrightness(self, indexStart, indexEnd):
        colors = self.colorData
        if indexEnd <= indexStart:
            return

        #Optimized quicksort doesn't play nice if there are only
        #two pixels to swap, so we treat this as a special case.
        if indexEnd - indexStart == 1:
            sumA = ColorSum(colors[indexStart])
            sumB = ColorSum(colors[indexEnd])
            if sumA <= sumB:
                return
            self.SwapPixels(indexStart, indexEnd)
            return

        pivotStart, pivotEnd = self.PartitionColorsByBrightness(
            indexStart, indexEnd)

        #Sort everything that wasn't part of this set of pivots.
        self.VisualQuickSortByBrightness(
            indexStart, pivotStart - 1)
        self.VisualQuickSortByBrightness(
            pivotEnd, indexEnd)

    """
    #================================================================
    Swap values both in our list and in Image object data.
    #================================================================
    """
    def SwapPixels(self, indexA, indexB):
        colors = self.colorData
        width = self.winfo_width() - 2

        if colors[indexA] == colors[indexB]:
            return

        colors[indexA], colors[indexB] = (
            colors[indexB], colors[indexA])

        x1 = indexA % width
        x2 = indexB % width
        y1 = math.floor(indexA / width)
        y2 = math.floor(indexB / width)
        self.draw.point((x1, y1), colors[indexA])
        self.draw.point((x2, y2), colors[indexB])

    """
    #================================================================
    Parition a list of colors based on a pivot value, displaying
    results live.

    Because we're dealing with a bitmapped image, there are likely
    to be swaths of repeated colors.  Vanilla quicksort is really
    bad at dealing with large chunks of repeated data, so this
    method is optimized to group all the values equal to the pivot
    together, and return two indices: where the group of homogeneous
    values start, and where they end.

    Since we're also guaranteed to be passed at least three values to
    sort, we can use the median of three colors chosen at random to
    boost the probability of partitioning efficiency.
    #================================================================
    """
    def PartitionColorsByHue(self, indexStart, indexEnd):
        colors = self.colorData
        indexCurrent = indexStart
        width = self.winfo_width() - 2

        #Use the median of three random values for a better pivot
        pivotIndex = sorted([
            random.randint(indexStart, indexEnd),
            random.randint(indexStart, indexEnd),
            random.randint(indexStart, indexEnd)])[1]
        self.SwapPixels(pivotIndex, indexEnd)
        colorPivot = colors[indexEnd]

        self.UpdateMarkers(
            indexStart, indexEnd, colors[indexCurrent])

        while indexCurrent <= indexEnd:
            #Only update graphics every line because Tk is slow
            if indexCurrent % width == 0:
                self.UpdateMarkers(
                    indexStart, indexEnd, colors[indexCurrent])

            #Move blue colors to the top of the canvas
            if colors[indexCurrent] < colorPivot:
                self.SwapPixels(indexCurrent, indexStart)
                indexStart += 1
                indexCurrent += 1
                continue

            #And move red colors to the bottom
            if colors[indexCurrent] > colorPivot:
                self.SwapPixels(indexCurrent, indexEnd)
                indexEnd -= 1
                continue

            #No work is required if the colors are the same
            indexCurrent += 1

        self.Refresh()
        return indexStart, indexCurrent

    """
    #================================================================
    While I could have incorporated the two sort algorthims into
    one with conditional statements, or crafting sorting methods
    and swapping out which once sort uses, I chose to keep them
    separate for simplicity and to maximise their speed of execution.
    #================================================================
    """
    def PartitionColorsByBrightness(self, indexStart, indexEnd):
        colors = self.colorData
        indexCurrent = indexStart
        width = self.winfo_width() - 2

        #Use the median of three random values for a better pivot
        pivotIndex = sorted([
            random.randint(indexStart, indexEnd),
            random.randint(indexStart, indexEnd),
            random.randint(indexStart, indexEnd)])[1]
        self.SwapPixels(pivotIndex, indexEnd)
        brightnessPivot = ColorSum(colors[indexEnd])

        self.UpdateMarkers(
            indexStart, indexEnd, colors[indexCurrent])

        while indexCurrent <= indexEnd:
            brightnessCurrent = ColorSum(colors[indexCurrent])

            #Only update graphics every line because Tk is slow
            if indexCurrent % width == 0:
                self.UpdateMarkers(
                    indexStart, indexEnd, colors[indexCurrent])

            #Move darker colors to the top of the canvas
            if brightnessCurrent < brightnessPivot:
                self.SwapPixels(indexCurrent, indexStart)
                indexStart += 1
                indexCurrent += 1
                continue

            #And move brighter colors to the bottom
            if brightnessCurrent > brightnessPivot:
                self.SwapPixels(indexCurrent, indexEnd)
                indexEnd -= 1
                continue

            #No work is required if they are equally bright
            indexCurrent += 1

        self.Refresh()
        return indexStart, indexCurrent

    """
    #================================================================
    Draw two little triangles on the canvas to show where
    quicksort is focusing it's attention. We can get away with this
    because we're cheating and the image data is sorted separately
    from the canvas itself.
    #================================================================
    """
    def UpdateMarkers(self, start, end, color):
        width = self.winfo_width() - 2
        
        x = start % width
        y = math.floor(start / width)
        self.DrawMarker(x, y, (*color, 255))
        
        x = end % width
        y = math.floor(end / width)
        self.DrawMarker(x, y, (*color, 255), True)

        self.Refresh()

    """
    #================================================================
    Create a mark on ths canvas to show where quicksort is working.
    #================================================================
    """
    def DrawMarker(self, x, y, color = WHITE, mirror = False):
        colorInverted = InvertColor(color)
        size = 6
        xOffset = 4
        if mirror == True:
            xOffset = -xOffset
        
        x1 = x
        y1 = y - size
        x2 = x + xOffset
        y2 = y
        x3 = x
        y3 = y + size

        self.create_polygon(
            (x1, y1, x2, y2, x3, y3),
            fill = ColorToHex(color),
            outline = ColorToHex(colorInverted))
        
        self.UpdateMethod()

    """
    #================================================================
    Try to open an image file.
    #================================================================
    """
    def LoadImage(self, filename):
        try:
            self.imageData = Image.open(filename)
        except:
            return False
        
        #New Image means we need a new Draw object.
        self.draw = ImageDraw.Draw(self.imageData, "RGBA")
        self.Refresh()
        return True

"""
#================================================================
"""

if __name__ == '__main__':
    print('This module is not meant to be run by itself.')
