#!/usr/bin/env python3

"""
#====================================================================

Hello reader,

Thank you for taking the time to look at my source code!

I've worked with Tk/perl before, and I've looked up the Python
style guidelines from https://peps.python.org/pep-0008/, but this is
my first ever Python script, so I'm sure there are conventions and
idioms that I'm as yet unaware of. If you have wisdom to offer me,
I will avail myself of it!

~Graham
RGrahamPreston42@gmail.com

#====================================================================
"""

#====================================================================
import math
import copy
import os
import sys
import re
import tkinter as tk
from PIL import Image, ImageTk

#My modules
import tkPixelCanvas as tkpc
import tkInputWidgets as tkiw
#====================================================================

"""
#====================================================================
Root class for use with Tk
#====================================================================
"""
class MainApp(tk.Frame):
    busyFlag = False
    showingAbout = False
    canvasSize = (150, 250)

    """
    #================================================================
    Initialization, this script accecepts command line parameters for
    the width and height of the PixelCanvas.
    #================================================================
    """
    def __init__(self, master):
        tk.Frame.__init__(self, master)
        self.master = master
        width, height = self.canvasSize
        
        if (len(sys.argv) > 1):
            if re.search("^\d+$", sys.argv[1]):
                width = int(sys.argv[1])
                if width < 20:
                    width = 20
                if width > 1400:
                    width = 1400
                    
        if (len(sys.argv) > 2):
            if re.search("^\d+$", sys.argv[2]):
                height = int(sys.argv[2])
                if height < 20:
                    height = 20
                if height > 900:
                    height = 900

        self.canvasSize = (width, height)

        self.ConfigWindow()
        self.ConfigWidgets()
        self.grid(row = 0, column = 0)

    """
    #================================================================
    Setup the basic settings for this project's window.
    #================================================================
    """
    def ConfigWindow(self):
        master = self.master

        self.menuBar = tk.Menu(master)
        master.config(menu = self.menuBar)
        master.title(
            "Programmer Preston's Perfectly Parsimonious Python Pixel "
            "Processor Project"
            )
        master.resizable(False, False)

    """
    #================================================================
    Build the GUI for this project. A lot of this mess got organized
    into the tkInputWidgets and tkPixelCanvas modules.
    #================================================================
    """
    def ConfigWidgets(self):
        master = self.master
        update = master.update

        #Set up the menu bar:
        self.menuFile = tk.Menu(self.menuBar, tearoff = 0)
        self.menuBar.add_cascade(
            label = 'File', menu = self.menuFile)
        self.menuFile.add_separator()
        self.menuFile.add_command(
            label = 'Save', command = self.ClickedSave)
        self.menuFile.add_separator()
        self.menuFile.add_command(
            label = 'Load', command = self.ClickedLoad)
        self.menuFile.add_separator()
        self.menuFile.add_command(
            label = 'Exit', command = master.quit)

        self.menuImage = tk.Menu(self.menuBar, tearoff = 0)
        self.menuBar.add_cascade(
            label = 'Image', menu = self.menuImage)
        self.menuImage.add_separator()
        self.menuImage.add_command(
            label = 'Fill', command = self.DoFill)
        self.menuImage.add_separator()
        self.menuImage.add_command(
            label = 'Visual Sort',
            command = lambda: self.DoSort(False))
        self.menuImage.add_separator()
        self.menuImage.add_command(
            label = 'Instant Sort',
            command = lambda: self.DoSort(True))

        self.menuHelp = tk.Menu(self.menuBar, tearoff = 0)
        self.menuBar.add_cascade(
            label = 'Help', menu = self.menuHelp)
        self.menuHelp.add_separator()
        self.menuHelp.add_command(
            label = 'About', command = self.ShowAbout)

        #Add the bespoke widgets from my modules:
        self.pixelCanvas = tkpc.PixelCanvas(
            self, update, self.canvasSize)
        self.colorInput = tkiw.ColorInput(self)
        self.fileInput = tkiw.FileInput(self)

        #A special widget to display polite messages to the user.
        self.labelPolite = tk.Label(
            self, text = "", font = "bold")

        #Grid all widgets
        self.colorInput.grid(row = 0, column = 0, sticky = tk.SE)
        self.fileInput.grid(row = 1, column = 0, sticky = tk.E)
        self.labelPolite.grid(row = 2, column = 0, sticky = tk.NE)
        self.pixelCanvas.grid(
            row = 0,
            rowspan = 3,
            padx = 20,
            column = 1,
            pady = 20,
            sticky = tk.NE)
        
        #Bind actions for user clicks on the PixelCanvas
        self.pixelCanvas.bind('<Button-1>', self.ClickedCanvas)
        self.pixelCanvas.bind('<B1-Motion>', self.ClickedCanvas)

    """
    #================================================================
    These methods are pretty heavy, so let's be polite about it!
    #================================================================
    """
    def SetBusyFlag(self, busy):
        if busy == True:
            self.busyFlag = True
            self.labelPolite.config(text = "Please be patient...")
            self.master.update()
            return
        
        self.busyFlag = False
        self.labelPolite.config(text = "")
        self.master.update()

    """
    #================================================================
    Fill any white areas on the canvas with random colors
    #================================================================
    """
    def DoFill(self):
        if self.busyFlag == True:
            return
        self.SetBusyFlag(True)

        self.pixelCanvas.FillEmptyPixels(
            self.colorInput.FillType())
        
        self.SetBusyFlag(False)

    """
    #================================================================
    Sort our canvas on a pixel by pixel basis? In tk? Well, I did it.
    #================================================================
    """
    def DoSort(self, instantFlag):
        if self.busyFlag == True:
            return
        self.SetBusyFlag(True)
        
        pc = self.pixelCanvas
        sortFlag = self.colorInput.SortType()

        #Let the fireworks begin!
        if instantFlag == False:
            pc.DoVisualQuickSort(sortFlag)
            self.SetBusyFlag(False)
            return
        
        #Boring instant sort
        if sortFlag == 1:
            sortedData = sorted(list(pc.imageData.getdata()))
        if sortFlag == 2:
            sortedData = sorted(
                list(pc.imageData.getdata()),
                key = tkpc.ColorSum)
        pc.imageData.putdata(sortedData)

        pc.Refresh()
        self.SetBusyFlag(False)

    """
    #================================================================
    Save the user's drawing to a .png file, making sure not to
    overwrite any exisitng files.
    #================================================================
    """
    def ClickedSave(self):
        if self.busyFlag == True:
            return
        
        saveFolder = "./P7 Saved Images/"
        filename = saveFolder + self.fileInput.Filename()

        try:
            os.mkdir(saveFolder)
        except:
            pass

        try:
            saveFile = open(filename, "x")
        except:
            saveFile = open(filename, "w")

        try:            
            self.pixelCanvas.imageData.save(saveFile.name, "PNG")
        except:
            self.labelPolite.config(text = "Error saving file!")
            return

    """
    #================================================================
    Planned for future versions, but not yet implimented
    #================================================================
    """
    def ClickedLoad(self):
        if self.busyFlag == True:
            return

        saveFolder = "./P7 Saved Images/"
        filename = saveFolder + self.fileInput.Filename()

        if self.pixelCanvas.LoadImage(filename) == False:
            self.labelPolite.config(text = "Error opening file!")

    """
    #================================================================
    Method called when the user clicks to draw on the canvas.
    #================================================================
    """
    def ClickedCanvas(self, event):
        if self.busyFlag == True:
            return
        
        color = self.colorInput.GetColor()
        size = self.colorInput.GetSize()
        self.pixelCanvas.DrawOval(event.x, event.y, color, size)

    """
    #================================================================
    All about me and this eloquently named project!
    I would have promoted this to it's own class but it was time to
    be done with this project.
    #================================================================
    """
    def ShowAbout(self):
        if self.showingAbout == True:
            return
        self.showingAbout = True

        self.aboutWindow = tk.Toplevel(self.master)
        self.aboutWindow.title("About Me")
        self.aboutWindow.resizable(False, False)

        try:
            self.splashImageData = Image.open("./GrahamAvatar.png")
        except:
            print("Failed to open about image!")
            return

        self.splashImageDataTk = ImageTk.PhotoImage(
            self.splashImageData)

        self.splashImage = tk.Label(
            self.aboutWindow,
            image = self.splashImageDataTk)
        self.splashImage.grid()

        self.splashImage.bind(
            '<Button-1>', self.DestroyAbout)
        self.aboutWindow.protocol(
            "WM_DELETE_WINDOW", self.DestroyAbout)
    
    def DestroyAbout(self, event = ""):
        self.aboutWindow.destroy()
        self.showingAbout = False
    
        

"""
#====================================================================
#All set!  Launch in a new window if being run as a standalone app.
#====================================================================
"""
if __name__ == '__main__':
    rootWindow = tk.Tk()
    mainApp = MainApp(rootWindow)
    rootWindow.mainloop()
